// $Id: CMidiFileWrite.cpp,v 1.2 2007/02/08 21:08:09 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CMidiFile.hpp"
using Exponent::Midi::CMidiFile;

//	===========================================================================
bool CMidiFile::writeFile(const CMidiSequence &sequence)
{
	// WRite the file header
	if (!writeFileHeader(m_stream, sequence))
	{
		return false;
	}

	// Write meta track data
	writeMetaTrack(m_stream, sequence);

	// Compute number of samples per tick
	const double ticksPerSample = computeTicksPerSample(sequence.getSampleRate(), sequence.getTempo());

	// Now write out the inidividual tracks
	for (long i = 0; i < sequence.getNumberOfTracks(); i++)
	{
		// Get the track
		const CMidiTrack *track = sequence.getTrackAtIndex(i);

		// If valid then write it out
		if (track)
		{
			writeTrack(m_stream, *track, ticksPerSample);
		}
	}

	// We are done :)
	return true;
}

//	===========================================================================
void CMidiFile::writeMetaTrack(CFileStream &stream, const CMidiSequence &sequence)
{
	// Write track header
	stream << 'M' << 'T' << 'r' << 'k';

	// Store the position of the stream for writing the size in at the end
	const long currentStreamPosition   = stream.getStreamPosition();
	stream.advanceStream((long)sizeof(unsigned long));
	
	// Write the meta events
	writeTimeSignature(stream, sequence.getTimeSignature());
	writeTempo(stream, sequence.getTempo());
	writeTrackEnd(stream);

	// Now compute the amount of data that we wrote
	const long finalStreamPosition = stream.getStreamPosition();
	const unsigned long streamSize = (unsigned long)(finalStreamPosition - currentStreamPosition);

	// Move back and write in the size, then reroll back to the end of the file
	stream.moveToStreamPosition(currentStreamPosition);
	stream << streamSize;
	stream.moveToStreamPosition(finalStreamPosition);
}

//	===========================================================================
void CMidiFile::writeTrackName(CFileStream &stream, const CString &name)
{
	// Write the header
	writeVariableLengthUnsignedLong(stream, 0);									// Name at the start of the track
	stream << (unsigned char)0xFF;												// Its a meta event
	stream << (unsigned char)CMidi::CMIDI_MIDIFILE_TRACK_NAME;					// Of type track name
	writeVariableLengthUnsignedLong(stream, name.getNumberOfCharacters() - 1);	// Size of midi meta event

	// Write the events
	const char *nameBuffer = name.getString();
	for (long i = 0; i < name.getNumberOfCharacters() - 1; i++)
	{
		stream << (unsigned char)nameBuffer[i];
	}
}

//	===========================================================================
void CMidiFile::writeTrackMidiChannel(CFileStream &stream, const long channel)
{
	// Write the header
	writeVariableLengthUnsignedLong(stream, 0);									// Channel at the start of the track
	stream << (unsigned char)0xFF;												// Its a meta event
	stream << (unsigned char)CMidi::CMIDI_MIDIFILE_TRACK_NUM;					// Of type track num
	writeVariableLengthUnsignedLong(stream, 1);									// Size of midi meta event
	stream << (unsigned char)channel;
}

//	===========================================================================
void CMidiFile::writeTempo(CFileStream &stream, const double bpm)
{
	// Write the header
	writeVariableLengthUnsignedLong(stream, 0);						// Tempo at the start of the track
	stream << (unsigned char)0xFF;									// Its a meta event
	stream << (unsigned char)CMidi::CMIDI_MIDIFILE_TEMPO;			// Of type tempo
	writeVariableLengthUnsignedLong(stream, 3);						// Size of midi meta event

	// Write the events
	const unsigned long tempoBase = static_cast<unsigned long>(60000000.0 / bpm);

	// Store the buffers, notice moving FF - Cos i didnt ;)
	const unsigned char buffer[3] = 
	{
		static_cast<unsigned char>((tempoBase & 0x00FF0000) >> 16),
		static_cast<unsigned char>((tempoBase & 0x0000FF00) >> 8),
		static_cast<unsigned char>((tempoBase & 0x000000FF) >> 0)
	};

	stream << buffer[0];
	stream << buffer[1];
	stream << buffer[2];
}

//	===========================================================================
void CMidiFile::writeTimeSignature(CFileStream &stream, const CTimeSignature ts)
{
	// Write the header
    writeVariableLengthUnsignedLong(stream, 0);							// TS at the start of the track
    stream << (unsigned char)0xFF;										// Its a meta event            
	stream << (unsigned char)CMidi::CMIDI_MIDIFILE_TIME_SIGNATURE;		// Of type time signature
    writeVariableLengthUnsignedLong(stream, 4);							// Meta event length

	// Bit shift about
	int numerator   = (int)ts.getNumerator();
	int denominator = (int)ts.getDenominator();

    unsigned char timeSigDenominator  = 0;
    while (denominator > 1) 
    {
        timeSigDenominator++;
        denominator >>= 1;
    }

	// Store the values
	stream << (unsigned char)numerator
		   << (unsigned char)timeSigDenominator
		   << (unsigned char)24										// Clocks per beat 
		   << (unsigned char)8;										// 32nd notes per 1/4 note
}

//	===========================================================================
bool CMidiFile::writeFileHeader(CFileStream &stream, const CMidiSequence &sequence)
{
	// Check open
	if (!stream.isStreamOpen())
	{
		return false;
	}

	// Failed cos no tracks
	if (sequence.getNumberOfTracks() <= 0)
	{
		return false;
	}

	// Read in the header
	stream << (unsigned char)'M' 
		   << (unsigned char)'T' 
		   << (unsigned char)'h' 
		   << (unsigned char)'d' 
		   << (unsigned long)6										// MThd always 6
		   << (unsigned short)1										// We write type 1 files
		   << (unsigned short)(sequence.getNumberOfTracks() + 1)	// Total number of tracks, plus our meta track
		   << (unsigned short)CMIDI_FILE_PPQ;						// We use 480PPQ, tempo based

	return true;
}

//	===========================================================================
bool CMidiFile::writeTrack(CFileStream &stream, const CMidiTrack &track, const double ticksPerSample)
{
	// Stream the header
	stream << 'M' << 'T' << 'r' << 'k';

	// Store the position of the stream for writing the size in at the end
	const long currentStreamPosition = stream.getStreamPosition();
	stream.advanceStream((long)sizeof(unsigned long));

	// Write in the midi channel event at the top of the file
	writeTrackMidiChannel(stream, track.getMidiChannel());
	writeTrackName(stream, track.getTrackName());

	// Some variables for us to use
    unsigned char status        = 0;
    unsigned char runningStatus = 0;
    long lastTimePosInTics      = 0;
    long nextTimePosInTics      = 0;
	const short channelMask 	= ((short)track.getMidiChannel())  % 16;

	// Loop through and add each event
	for (long i = 0; i < track.getNumberOfEvents(); i++)
	{
		// get the event
		const CMidiEvent *event = track.getEventAtIndex(i);

		// Check its valid
		if (event == NULL)
		{
			continue;
		}

		// If its a writable event
		if (isWritableEvent(*event))
		{
			// Write the delta time
			//ticksPerSample
			double timeTag    = ((double)event->getTimeDelta()) * ticksPerSample;
            nextTimePosInTics = (long)timeTag;
			writeVariableLengthUnsignedLong(stream, (nextTimePosInTics - lastTimePosInTics));
			lastTimePosInTics = nextTimePosInTics;

			if (event->isNoteOn())
			{
				status = (unsigned char)(CMidi::CMIDI_NOTE_ON + channelMask);
				if (status != runningStatus)
				{
					stream << status;
					runningStatus = status;
				}

				// Stream out the notes
				stream << (unsigned char)event->getValue();
				stream << (unsigned char)event->getSubValue();
			}
			else if (event->isNoteOff())
			{
				status = (unsigned char)(CMidi::CMIDI_NOTE_OFF + channelMask);
				if (status != runningStatus)
				{
					stream << status;
					runningStatus = status;
				}

				// Stream out the notes
				stream << (unsigned char)event->getValue();
				stream << (unsigned char)event->getSubValue();
			}
			else if (event->isControlChange())
			{
				status = (unsigned char)(CMidi::CMIDI_CONTROL_CHANGE + channelMask);
				if (status != runningStatus)
				{
					stream << status;
					runningStatus = status;
				}

				// Stream out the notes
				stream << (unsigned char)event->getValue();
				stream << (unsigned char)event->getSubValue();
			}
			else if (event->isPitchBend())
			{
				status = (unsigned char)(CMidi::CMIDI_PITCH_BEND + channelMask);
				if (status != runningStatus)
				{
					stream << status;
					runningStatus = status;
				}

				// This gives us a pitch bend of somewhere between -1 and +1
				const double pitchBend = event->getPitchBend();
				stream << (unsigned char)0;				// NO fine increment

				if (pitchBend == 0)
				{
					stream << (unsigned char)64;		// NO pitch bend
				}
				else
				{
					const long position = (long)ceil(64 + ((pitchBend > 0) ? pitchBend : (1.0 - (-pitchBend)) * (double)(127 - 64)));
					stream << (unsigned char)position;
				}
			}
			else if (event->isAllNotesOff())
			{
				status = (unsigned char)(CMidi::CMIDI_CONTROL_CHANGE + channelMask);
				if (status != runningStatus)
				{
					stream << status;
					runningStatus = status;
				}

				// Stream out the notes
				stream << (unsigned char)CMidi::CMIDI_ALL_NOTES_OFF;
				stream << (unsigned char)0;
			}
		}
	}

	// Write the track ending
	writeTrackEnd(stream);

	// Now compute the amount of data that we wrote
	const long finalStreamPosition = stream.getStreamPosition();
	const unsigned long streamSize = (unsigned long)(finalStreamPosition - currentStreamPosition);

	// Move back and write in the size, then reroll back to the end of the file
	stream.moveToStreamPosition(currentStreamPosition);
	stream << streamSize;
	stream.moveToStreamPosition(finalStreamPosition);

	return true;
}

//	===========================================================================
void CMidiFile::writeTrackEnd(CFileStream &stream)
{
	unsigned char buffer[4] = { 0x00, 0xFF, 0x2F, 0x00 };
	stream.writeDataToStream(buffer, 4 * sizeof(unsigned char));
}

//	===========================================================================
void CMidiFile::writeVariableLengthUnsignedLong(CFileStream &stream, const unsigned long value)
{
	// Some temporary variables
	unsigned char buffer[10];
    int length			  = 0; 
	unsigned long myValue = value;

	// Zero the buffer
	memset(buffer, 0, 10 * sizeof(unsigned char));

    do 
	{
        buffer[length] = ((length > 0) ? 0x80 : 0x00) | (unsigned char)(myValue & 0x7F);
        length++;
        myValue >>= 7;
    }
	while (myValue > 0);

    while (length) 
	{
        length--;
		const unsigned char character = buffer[length];
        stream << character;
    }
}

//	===========================================================================
bool CMidiFile::isWritableEvent(const CMidiEvent &event)
{
	return (event.isNoteOn() || event.isNoteOff() || event.isControlChange()	|| event.isPitchBend() || event.isAllNotesOff());
}

//	===========================================================================
double CMidiFile::computeTicksPerSample(const double sampleRate, const double bpm)
{
	const static double bpsDivisor = 0.016666666666666666666666666666667;	// 1 / 60
	double beatsPerSecond = bpm * bpsDivisor;
	return 1.0 / (sampleRate / beatsPerSecond / (double)CMIDI_FILE_PPQ);
}